<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\MakeAndModel;
use App\Models\SoftwareLicense\PaymentMethod;
use App\Models\SoftwareLicense\SoftwareLicense;
use App\Models\SoftwareLicense\SoftwareLicenseCategory;
use App\Models\SoftwareLicense\SoftwareLicenseDocument;
use App\Models\SoftwareLicense\SoftwareLicenseHistory;
use App\Models\SoftwareLicense\SoftwareLicenseKey;
use App\Models\SoftwareLicense\SoftwareLicenseManufacturer;
use App\Models\SoftwareLicense\SoftwareLicenseSubscription;
use App\Models\SoftwareLicense\SoftwareLicenseUser;
use App\Models\Vendor;
use App\Repositories\SoftwareLicense\LicenseRepository;
use App\Services\SoftwareLicense\LicenseHistoryService;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Faker\Factory as Faker;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;

class TQT195SoftwareAssets extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'SotwareAssets:update';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update SotwareAssets on TQT-195';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(private LicenseRepository $licenseRepo, private LicenseHistoryService $licenseHistoryService)
    {
        parent::__construct();
        $this->licenseRepo = $licenseRepo;
        $this->licenseHistoryService = $licenseHistoryService;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {

        $softwareApplications = [
            "Operating systems" => [
                "Windows" => [
                    "vendor" => "Microsoft",
                    "license" => "Paid"
                ],
                "MacOS" => [
                    "vendor" => "Apple",
                    "license" => "Paid"
                ],
                "Linux" => [
                    "vendor" => "Open source community",
                    "license" => "Free"
                ],
                "ChromeOS" => [
                    "vendor" => "Google",
                    "license" => "Free"
                ]
            ],
            "Productivity software" => [
                "Microsoft Office" => [
                    "vendor" => "Microsoft",
                    "license" => "Paid"
                ],
                "Google Workspace" => [
                    "vendor" => "Google",
                    "license" => "Paid"
                ],
                "Adobe Creative Suite" => [
                    "vendor" => "Adobe",
                    "license" => "Paid"
                ]
            ],
            "Graphic design software" => [
                "Adobe Photoshop" => [
                    "vendor" => "Adobe",
                    "license" => "Paid"
                ],
                "GIMP" => [
                    "vendor" => "Open source community",
                    "license" => "Free"
                ],
                "Canva" => [
                    "vendor" => "Canva",
                    "license" => "Freemium"
                ]
            ],
            "Video editing software" => [
                "Adobe Premiere Pro" => [
                    "vendor" => "Adobe",
                    "license" => "Paid"
                ],
                "Final Cut Pro" => [
                    "vendor" => "Apple",
                    "license" => "Paid"
                ],
                "iMovie" => [
                    "vendor" => "Apple",
                    "license" => "Free"
                ]
            ],
            "Web browsers" => [
                "Google Chrome" => [
                    "vendor" => "Google",
                    "license" => "Free"
                ],
                "Mozilla Firefox" => [
                    "vendor" => "Mozilla",
                    "license" => "Free"
                ],
                "Microsoft Edge" => [
                    "vendor" => "Microsoft",
                    "license" => "Free"
                ],
                "Safari" => [
                    "vendor" => "Apple",
                    "license" => "Free"
                ]
            ],
            "Email clients" => [
                "Microsoft Outlook" => [
                    "vendor" => "Microsoft",
                    "license" => "Paid"
                ],
                "Gmail" => [
                    "vendor" => "Google",
                    "license" => "Free"
                ],
                "Apple Mail" => [
                    "vendor" => "Apple",
                    "license" => "Free"
                ]
            ],
            "Instant messaging software" => [
                "Slack" => [
                    "vendor" => "Slack",
                    "license" => "Freemium"
                ],
                "Microsoft Teams" => [
                    "vendor" => "Microsoft",
                    "license" => "Freemium"
                ],
                "Skype" => [
                    "vendor" => "Microsoft",
                    "license" => "Freemium"
                ]
            ],

            "Music players" => [
                "iTunes" => ["vendor" => "Apple", "license" => "Free"],
                "Spotify" => ["vendor" => "Spotify", "license" => "Freemium"],
                "Winamp" => ["vendor" => "Nullsoft", "license" => "Free"]
            ],
            "Gaming software" => [
                "Steam" => ["vendor" => "Valve", "license" => "Free"],
                "Epic Games Store" => ["vendor" => "Epic Games", "license" => "Free"],
                "Origin" => ["vendor" => "Electronic Arts", "license" => "Free"]
            ],
            "Antivirus software" => [
                "Norton" => ["vendor" => "NortonLifeLock", "license" => "Paid"],
                "McAfee" => ["vendor" => "McAfee", "license" => "Paid"],
                "Avast" => ["vendor" => "Avast", "license" => "Freemium"]
            ],
            "Programming languages" => [
                "Java" => ["vendor" => "Oracle", "license" => "Free"],
                "Python" => ["vendor" => "Python Software Foundation", "license" => "Free"],
                "PHP" => ["vendor" => "The PHP Group", "license" => "Free"],
            ],
            "Database management systems" => [
                "Oracle" => ["vendor" => "Oracle", "license" => "Paid"],
                "MySQL" => ["vendor" => "Oracle", "license" => "Free"],
                "Microsoft SQL Server" => ["vendor" => "Microsoft", "license" => "Paid"],
                "PostgreSQL" => ["vendor" => "PostgreSQL Global Development Group", "license" => "Free"]
            ],
            "Content management systems" => [
                "WordPress" => ["vendor" => "WordPress Foundation", "license" => "Free"],
                "Drupal" => ["vendor" => "Drupal Association", "license" => "Free"],
                "Joomla" => ["vendor" => "Joomla Project", "license" => "Free"],
                "Magento" => ["vendor" => "Adobe", "license" => "Free"]
            ],
            "Virtualization software" => [
                "VMware" => ["vendor" => "VMware", "license" => "Paid"],
                "VirtualBox" => ["vendor" => "Oracle", "license" => "Free"],
                "Hyper-V" => ["vendor" => "Microsoft", "license" => "Paid"]
            ],

            "Cloud computing platforms" => [
                "Amazon Web Services" => [
                    "vendor" => "Amazon",
                    "license" => "Paid"
                ],
                "Microsoft Azure" => [
                    "vendor" => "Microsoft",
                    "license" => "Paid"
                ],
                "Google Cloud Platform" => [
                    "vendor" => "Google",
                    "license" => "Paid"
                ]
            ],
            "Project management software" => [
                "Trello" => [
                    "vendor" => "Atlassian",
                    "license" => "Freemium"
                ],
                "Asana" => [
                    "vendor" => "Asana",
                    "license" => "Freemium"
                ],
                "Basecamp" => [
                    "vendor" => "Basecamp",
                    "license" => "Freemium"
                ]
            ],
            "Customer relationship management software" => [
                "Salesforce" => [
                    "vendor" => "Salesforce",
                    "license" => "Paid"
                ],
                "HubSpot" => [
                    "vendor" => "HubSpot",
                    "license" => "Freemium"
                ],
                "Zoho CRM" => [
                    "vendor" => "Zoho",
                    "license" => "Freemium"
                ]
            ],
            "Enterprise resource planning software" => [
                "SAP" => [
                    "vendor" => "SAP",
                    "license" => "Paid"
                ],
                "Oracle ERP" => [
                    "vendor" => "Oracle",
                    "license" => "Paid"
                ],
                "Microsoft Dynamics 365" => [
                    "vendor" => "Microsoft",
                    "license" => "Paid"
                ]
            ],
            "Financial software" => [
                "QuickBooks" => [
                    "vendor" => "Intuit",
                    "license" => "Paid"
                ],
                "Xero" => [
                    "vendor" => "Xero",
                    "license" => "Paid"
                ],
                "FreshBooks" => [
                    "vendor" => "FreshBooks",
                    "license" => "Paid"
                ]
            ],
            "Human resources software" => [
                "BambooHR" => [
                    "vendor" => "BambooHR",
                    "license" => "Paid"
                ],
                "Workday" => [
                    "vendor" => "Workday",
                    "license" => "Paid"
                ],
                "ADP" => [
                    "vendor" => "ADP",
                    "license" => "Paid"
                ]
            ]
        ];

        Schema::disableForeignKeyConstraints();
        SoftwareLicenseDocument::truncate();
        SoftwareLicenseHistory::truncate();
        SoftwareLicenseKey::truncate();
        SoftwareLicenseUser::truncate();
        SoftwareLicense::truncate();
        SoftwareLicenseCategory::truncate();
        SoftwareLicenseManufacturer::truncate();
        Schema::enableForeignKeyConstraints();


        $manufacturers = collect($softwareApplications)->flatten(1)->pluck('vendor')->unique()->toArray();

        $insertData = [];

        foreach ($manufacturers as $manufacturer) {
            $insertData[] = ['name' => $manufacturer, 'created_at' => Carbon::now(), 'updated_at' => Carbon::now()];
        }

        SoftwareLicenseManufacturer::insert($insertData);


        $categoryData = [];

        foreach (array_keys($softwareApplications) as $category) {
            $categoryData[] = ['name' => $category, 'created_at' => Carbon::now(), 'updated_at' => Carbon::now()];
        }

        SoftwareLicenseCategory::insert($categoryData);

        foreach ($softwareApplications as $category => $software) {
            foreach ($software as $name => $details) {
                $faker = Faker::create();
                $licenseTypes = array_keys(config('software-license.license_types'));
                $licenseType = $licenseTypes[random_int(0, count($licenseTypes) - 1)];
                $types = array_keys(config('software-license.types'));
                $type = $types[random_int(0, count($types) - 1)];
                $autoRenewals = array_keys(config('software-license.auto_renewal'));
                $autoRenewal = $autoRenewals[random_int(0, count($autoRenewals) - 1)];
                $billingCycles = array_keys(config('software-license.billing_cycles'));
                $status = array_keys(config('software-license.licence_status'));
                $licenseKeyTypes = array_keys(array_diff_key(config('software-license.license_key_types'), ['no_license' => '']));
                $licenseKeyType = $licenseKeyTypes[random_int(0, count($licenseKeyTypes) - 1)];
                $keyUsageTypes = array_keys(config('software-license.key_usage_types'));
                $keyUsageType = $licenseKeyType != 'no_license' ? $keyUsageTypes[random_int(0, count($keyUsageTypes) - 1)] : '';
                $usersPerKey = $keyUsageType == 'limited_usage' ? random_int(2, 6) : '';

                $startDate = $faker->dateTimeBetween('-2 years', 'now')->format('m/d/Y');

                $softwareLicenseSubscriptionId = SoftwareLicenseSubscription::inRandomOrder()->value('id');
                // // $notificationDates = array_keys(config('software-license.notification_period'));
                $license = SoftwareLicense::create([
                    'software_license_category_id' => SoftwareLicenseCategory::where('name', $category)->first()->id,
                    'type' => $type,
                    'software_license_manufacturer_id' => SoftwareLicenseManufacturer::where('name', $details['vendor'])->first()->id,
                    'vendor_id' => Vendor::inRandomOrder()->value('id'),
                    'name' => $name,
                    'po_number' => $faker->numerify('########'),
                    'order_number' => $faker->numerify('########'),
                    'license_type' => $licenseType,
                    'billing_cycle' => $billingCycles[random_int(0, count($billingCycles) - 1)],
                    'start_date' => convert_to_db_date($startDate),
                    'software_license_subscription_id' => $licenseType == 'subscription' ? $softwareLicenseSubscriptionId : '',
                    'renewal_date' => $licenseType == 'subscription' ? $this->getRenewalDate($startDate, $softwareLicenseSubscriptionId, $licenseType) : NULL,
                    'auto_renewal' => $autoRenewal,

                    'license_keys_added' => $faker->numberBetween(0, 10),
                    'payment_method_id' => PaymentMethod::inRandomOrder()->value('id'),
                    'notes' => $faker->text(),
                    'status' => $status[random_int(0, count($status) - 1)],

                    'license_key_type' => $details['license'] == 'Paid' ? $licenseKeyType : 'no_license',
                    'key_usage_type' => $keyUsageType,
                    'users_allowed_for_key' => $usersPerKey,

                    'licenses_purchased' => $licenseKeyType == 'user_license' ? $faker->numberBetween(2, 10) : '',

                ]);

                if ($license->license_key_type != 'no_license') {
                    $this->createLicenseKeys($license);
                }

                $this->createLicenseHistory($license);
            }
        }

        $this->info("Created licenses");
    }

    public function createLicenseKeys($license, $action = 'license_key_added')
    {
        if ($license->license_key_type == 'user_license') {
            for ($i = 0; $i < $license->licenses_purchased; $i++) {
                $this->storeKeys($license, $action);
            }
        } else {
            $this->storeKeys($license, $action);
        }

        return true;
    }

    public function storeKeys($license, $action)
    {
        $keyArray = [];
        $faker = Faker::create();
        $keyArray = [
            'software_license_id' => $license->id,
            'license_key' => $faker->regexify('[A-Z]{3}[0-4]{3}[a-z]{3}'),
            'key_usage_type' => $license->key_usage_type,
            'users_allowed_for_key' => $license->key_usage_type  && in_array($license->key_usage_type, ['limited_usage']) ? $license->users_allowed_for_key : ($license->key_usage_type == 'unlimited_usage' ? 'U' : 1),
            'no_of_users' => ($license->key_usage_type == 'unlimited_usage') ? 'U' : 0,
        ];
        $licenseKey = SoftwareLicenseKey::create($keyArray);
        $historyData = $this->licenseHistoryService->getHistoryDataForKeyAdded($action, $licenseKey);

        $this->licenseRepo->addLicenseHistory($historyData);
    }

    public function createLicenseHistory($license)
    {
        $description = $this->licenseHistoryService->getStatusDescriptionForCreated($license);

        $this->licenseRepo->addLicenseHistory([
            'software_license_id' => $license->id,
            'action' => 'created',
            'description' => $description,
            'user_id' => Auth::id()
        ]);

        return true;
    }

    public function getRenewalDate($startDate, $subscriptionLength, $licenseType)
    {
        $noRenwalDateSubcriptions = ['perpetual'];

        if (!$startDate || !$subscriptionLength || in_array($licenseType, $noRenwalDateSubcriptions)) {
            return null;
        }
        $subscriptionLengthData = SoftwareLicenseSubscription::find($subscriptionLength);

        $renewalDate = ($subscriptionLengthData->duration_type == 'Years' || $subscriptionLengthData->duration_type == 'Year') ? addYears($startDate, $subscriptionLengthData->duration) : addMonths($startDate, $subscriptionLengthData->duration);
        // dd($subscriptionLengthData,$subscriptionLength,$renewalDate,$startDate);
        return convert_to_db_date($renewalDate);
    }
}
