<?php

namespace App\Console\Commands;

use App\Models\AirWatch;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Chromebook;
use App\Models\IntuneComputer;
use App\Models\JamfComputer;
use App\Models\KandjiDevices;
use App\Models\MakeAndModel;
use App\Models\Manufacturer;
use App\Models\MobileIron;
use App\Models\TechnicalSpecs;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Factory;
use Illuminate\Support\Facades\DB;
use Faker\Factory as Faker;

class TQT195DiscoveryTools extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'DiscoveryTools:update';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Discovery Tools updates on TQT-195';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        AirWatch::truncate();
        Chromebook::truncate();
        IntuneComputer::truncate();
        JamfComputer::truncate();
        KandjiDevices::truncate();
        MobileIron::truncate();

        $allManufacturers = Manufacturer::pluck('slug')->toArray();

        AirWatch::factory()->count(200)->create();

        $this->makeInactiveDiscrepancyDevices(new AirWatch, $allManufacturers);
        $this->makeOnhandDevices(new AirWatch, $allManufacturers);

        $this->info("Workspace ONE created.");


        sleep(1);

        Chromebook::factory()->count(200)->create();

        $this->createChromebookMakeModels();
        $this->makeInactiveDiscrepancyDevices(new Chromebook, $allManufacturers, 'last_sync');
        $this->makeOnhandDevices(new Chromebook, $allManufacturers, 'last_sync');

        $this->info("Chrome devices created.");

        sleep(1);


        IntuneComputer::factory()->count(200)->create();

        $this->makeInactiveDiscrepancyDevices(new IntuneComputer, ['dell', 'lenovo'], 'checkin_date');
        $this->makeOnhandDevices(new IntuneComputer, ['dell', 'lenovo'], 'checkin_date');

        $this->info("Intune computers created.");

        sleep(1);


        JamfComputer::factory()->count(200)->create();

        $this->makeInactiveDiscrepancyDevices(new JamfComputer, ['apple'], 'checkin_date');
        $this->makeOnhandDevices(new JamfComputer, ['apple'], 'checkin_date');

        $this->info("Jamf computers created.");

        sleep(1);


        KandjiDevices::factory()->count(200)->create();

        $this->makeInactiveDiscrepancyDevices(new KandjiDevices, ['apple'], 'last_checkin');
        $this->makeOnhandDevices(new KandjiDevices, ['apple'], 'last_checkin');

        $this->info("Kandji devices created.");

        sleep(1);


        MobileIron::factory()->count(200)->create();

        $this->makeInactiveDiscrepancyDevices(new MobileIron, $allManufacturers, 'last_checkin');
        $this->makeOnhandDevices(new MobileIron, $allManufacturers, 'last_checkin');

        $this->info("MobileIrons created.");

        session()->forget('airwatchsession');
        session()->forget('intunesession');
        session()->forget('jamfsession');
        session()->forget('kandjisession');
        session()->forget('mobsession');
        session()->forget('chromesession');
    }

    /**
     * Devices for user descrepancy and not checking in managed deployed
     * @param mixed $model
     * @param string $lastSeenField
     * 
     * @return [type]
     */
    public function makeInactiveDiscrepancyDevices($model, $manufacturers, $lastSeenField = 'last_seen')
    {
        $mdmDevices = $model->limit(40)->get();
        $faker = Faker::create();
        $statusId   = AssetStatus::whereIn('slug', ['assigned', 'loaned'])->get()->pluck('id');
        $statusAssets = Asset::whereIn('asset_status_id', $statusId)
            ->whereHas('assetType', fn($query) => $query->whereIn('slug', ['laptop', 'desktop', 'mobile_phone', 'tablet']))
            ->whereHas(
                'makeAndModel',
                fn($query) => $query
                    ->whereHas('manufacturer', fn($query) => $query->whereIn('slug', $manufacturers))
            )
            ->hasNoDiscoveryAssets()
            ->limit(40)->get();

        $lastdatetime = $faker->dateTimeBetween('-1 year', '-30 days');
        $lastSeenDatetime = $lastdatetime->format('Y-m-d\TH:i:s.u');

        if ($lastSeenField == 'checkin_date') {
            $lastdatetime = $faker->dateTimeBetween('-1 year', '-30 days');
            $lastSeenDatetime = $lastdatetime->format("Y-m-d H:i:s");
            $lastSeenDatetime = Carbon::createFromFormat("Y-m-d H:i:s", $lastSeenDatetime)->format('Y-m-d\TH:i:s\Z');
        }

        if ($model instanceof MobileIron) {
            $lastdatetime = $faker->dateTimeBetween('-1 year', '-30 days');
            $lastSeenDatetime = $lastdatetime->format("Y-m-d H:i:s");
            $lastSeenDatetime = Carbon::createFromFormat("Y-m-d H:i:s", $lastSeenDatetime)->timestamp * 1000;
        }

        $i = 0;
        foreach ($mdmDevices as $device) {
            if ($i < 20) {
                $device->update([
                    'asset_id' => $statusAssets[$i]->id,
                    'serial_no' => $statusAssets[$i]->serial_no,
                    $lastSeenField => $lastSeenDatetime
                ]);
            } else {
                $device->update([
                    'asset_id' => $statusAssets[$i]->id,
                    'serial_no' => $statusAssets[$i]->serial_no,
                ]);
            }
            $i++;
        }
    }

    /**
     * Devices for checking in on hand
     * @param mixed $model
     * @param string $lastSeenField
     * 
     * @return [type]
     */
    public function makeOnhandDevices($model, $manufacturers, $lastSeenField = 'last_seen')
    {
        $mdmDevices = $model->skip(40)->limit(20)->get();
        $faker = Faker::create();
        $statusId   = AssetStatus::onHandStatuses()->get()->pluck('id');
        $statusAssets = Asset::whereIn('asset_status_id', $statusId)
            ->whereHas('assetType', fn($query) => $query->whereIn('slug', ['laptop', 'desktop', 'mobile_phone', 'tablet']))
            ->whereHas('makeAndModel', function ($query) use ($statusId) {
                $query->where('name', 'like', '%Chromebook%')->orWhere('name', 'like', '%Google%');
            })
            ->hasNoDiscoveryAssets()->limit(20)->get();

        $assetLastSeen = $faker->dateTimeBetween('-60 days', '-30 days');
        $assetLastSeenDatetime = $assetLastSeen->format('Y-m-d\TH:i:s.u');

        $lastdatetime = $faker->dateTimeBetween('-30 days', 'now');
        $lastSeenDatetime = $lastdatetime->format('Y-m-d\TH:i:s.u');


        if ($lastSeenField == 'checkin_date') {
            $lastdatetime = $faker->dateTimeBetween('-30 days', 'now');
            $lastSeenDatetime = $lastdatetime->format("Y-m-d H:i:s");
            $lastSeenDatetime = Carbon::createFromFormat("Y-m-d H:i:s", $lastSeenDatetime)->format('Y-m-d\TH:i:s\Z');
        }


        $i = 0;
        foreach ($mdmDevices as $device) {

            if (isset($statusAssets[$i])) {
                $device->update([
                    'asset_id' => $statusAssets[$i]->id,
                    'serial_no' => $statusAssets[$i]->serial_no,
                    $lastSeenField => $lastSeenDatetime
                ]);

                $statusAssets[$i]->update(['last_status_update' => $assetLastSeenDatetime]);
            }

            $i++;
        }
    }

    /**
     * Creates chromebooks make and models
     * @return [type]
     */
    public function createChromebookMakeModels()
    {

        $chromebooks = [
            'AOpen Chromebox Commercial 2',
            'ASUS Chromebook Flip C436FA',
            'AOpen Chromebox Commercial',
            'AOpen Chromebox Mini',
            'Google Pixelbook',
            'AOpen Chromebox Mini',
            'Samsung Galaxy Chromebook',
            'Acer Chromebook Spin 513, CP513-1H/1HL, R841T/LT',
            'Samsung Galaxy Chromebook 2',
            'ASUS Chromebook Flip CX5 (CX5400)',
            'ASUS Chromebox 4',
            'Google Pixelbook Go'
        ];


        $assetTypeId = AssetType::where('slug', 'laptop')->first()->id;

        foreach ($chromebooks as $chromebook) {
            $faker = Faker::create();

            $model = $chromebook;

            $manufacturer = Manufacturer::firstOrCreate(
                ['slug' => str_slug(explode(' ', $model)[0], '_')],
                ['name' => explode(' ', $model)[0], 'slug' => str_slug(explode(' ', $model)[0], '_')]
            )->id;

            $makeAndModel = MakeAndModel::firstOrCreate(
                ['slug' => str_slug($model, '_')],
                ['manufacturer_id' => $manufacturer, 'name' => $model, 'slug' => str_slug($model, '_'), 'asset_type_id' => $assetTypeId]
            );

            $specs = ['M1/64GB/4TB', '3.0GHz/8GB/256GB', '2.7GHz/16GB/128GB', '2.6GHz/8GB/1TB', '3.0GHz/16GB/512GB'];
            $details = $faker->randomElement($specs);

            $technicalSpecs = TechnicalSpecs::updateOrCreate(
                ['make_and_model_id' => $makeAndModel->id],
                ['make_and_model_id' => $makeAndModel->id, 'details' => $details, 'original_value' => '100', 'new_value' => 0, 'life_of_asset' => 4, 'status' => 1]
            );

            $statusId    = $faker->randomElement(AssetStatus::whereIn('slug', ['loaned', 'assigned'])->get()->pluck('id')->toArray());

            $chromeAssetCount = Asset::where(['make_and_model_id' => $makeAndModel->id])->count();
            if ($chromeAssetCount < 30) {
                Asset::factory()->count(30)->create([
                    'asset_type_id' => $makeAndModel->asset_type_id,
                    'asset_status_id' => $statusId,
                    'make_and_model_id' => $makeAndModel->id,
                    'technical_spec_id' => $technicalSpecs->id,
                ]);
            }
        }
    }
}
