<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use App\Events\UpdateShiShipmentStatus;
use Illuminate\Console\Command;
use App\Services\Integrations\Asn\AsnShiIntegration;
use App\Services\ShipmentTracking\ShiShipmentService;
use App\Models\CronJob;
use App\Notifications\ShiImportNotification;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Log;

class ShiImport extends Command
{
    protected $service;
    protected $shipmentService;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'ShiImport:import';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import the ASN SHI assets through API';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(AsnShiIntegration $service, ShiShipmentService $shipmentService)
    {
        parent::__construct();
        $this->service = $service;
        $this->shipmentService = $shipmentService;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $execStartTime  = Carbon::now();
        $dateFrom       = "2021-01-01 00:00:00";
        $dateTo         = Carbon::now();

        $loggedData = CronJob::where('action', 'asn_shi')->orderBy('created_at', 'desc')->first();
        if ($loggedData) {
            $dateFrom = Carbon::parse($loggedData->date_to);
        }

        if (!$this->service->checkConnection()) {
            $this->sendNotification(false, "SHI Import connection error");
            return $this->info("connection_error");
        }

        $totalImported = $this->service->importShiDetails($dateFrom);
        $this->updateImportLog($dateFrom, $dateTo, $execStartTime);
        $message = "SHI Import: " . $totalImported . " assets imported";

        event(new UpdateShiShipmentStatus);
        $this->sendNotification(true, $message);

        // Event to add sync history(sync_name, sync_status, notes).
        event(new IntegrationSyncHistory('asn-shi'));

        return $this->info($message);
    }


    /**
     * Sends a notification message via Slack.
     *
     * @param string $message The message content to be sent via Slack.
     *
     * @return bool
     */
    public function sendNotification($message)
    {
        try {
            Notification::route('slack', config('error-notification-mails.slack'))->notify(new ShiImportNotification($message));
        } catch (Exception $e) {
            Log::error($e->getMessage());
        }
    }

    /**
     * Update the cron table with execution time details
     *
     * @param Date $dateFrom         import data start date
     * @param Date $dateTo            Import data end date
     * @param Date $execStartTime    Import start time
     */
    public function updateImportLog($dateFrom, $dateTo, $execStartTime)
    {
        $cronData = [
            'action'        => 'asn_shi',
            'started_at'    => $execStartTime,
            'ended_at'      => Carbon::now(),
            'date_from'     => $dateFrom,
            'date_to'       => $dateTo,
        ];

        CronJob::create($cronData);
    }
}
