<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\Integrations\RevivnIntegrationService;
use App\Services\Revivn\RevivnSyncService;
use Illuminate\Support\Facades\Notification;
use App\Notifications\Slack;

class RevivnDataSync extends Command
{
    protected $signature = 'RevivnData:sync';
    protected $description = 'Sync Revivn assets and update asset status to end_of_life_disposed status';

    /**
     * Constructor for the class.
     *
     * @param RevivnIntegrationService $apiService The RevivnIntegrationService object.
     * @param RevivnSyncService $syncService The RevivnSyncService object.
     */
    public function __construct(protected RevivnIntegrationService $apiService, protected  RevivnSyncService $syncService)
    {
        parent::__construct();
    }

    /**
     * Handle the execution of the cron job.
     *
     * Checks API connection, fetches Revivn pickup and assets data,
     * and provides informational messages.
     *
     * @return void
     */
    public function handle()
    {
        if (!$this->apiService->checkApiConnection()) {
            $this->error("\nFailed to connect\n");

            $this->sendNotification(false);
            return;
        }

        $this->info("Cron Started");
        $this->fetchRevivnPickupData();
        $this->fetchRevivnAssetsData();

        $this->sendNotification(true);
        $this->info("Cron Completed");
    }

    /**
     * Fetches Revivn assets data and processes it.
     *
     * Retrieves assets data from the Revivn service, considering pagination,
     * and delegates the creation of Revivn data to the service.
     *
     * @return bool Returns true after successfully processing all pages of assets data.
     */
    protected function fetchRevivnAssetsData()
    {
        $response = $this->apiService->getAssets();

        $totalPages = $response->meta->total_pages;

        for ($i = 1; $i <= $totalPages; $i++) {
            $data = $this->apiService->getAssets($i);
            $assets = $data->assets;
            $this->syncService->createRevivnAssetData($assets);
        }

        return true;
    }

    /**
     * Fetches the Revivn pickup data and call function to create and 
     * updates it in the database.
     *
     * @return boolean true on success.
     */
    protected function fetchRevivnPickupData()
    {
        $pickupData = $this->apiService->getPickups();
        if (empty($pickupData->pickups)) {
            return true;
        }

        foreach ($pickupData->pickups as $pickup) {
            $this->syncService->createPickupData($pickup);
        }
        return true;
    }

    /**
     *  It sends a notification to a Slack channel using the Slack webhook URL
     * 
     * @param bool status true or false
     */
    public function sendNotification(bool $status)
    {
        if ($status) {
            $message = 'Revivn sync completed.';
        } else {
            $message = 'Revivn sync failed.';
        }

        if (config('teqtivity-notification-mails.slack')) {
            Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));
        }
    }
}
