<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CronJob;
use App\Notifications\Slack;
use App\Notifications\SyncNotification;
use App\Services\Asset\AsnPresidioIntegration;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class PresidioImport extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'presidio:import';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import the Presidio API data';

    protected $presidioIntegrationService;

    /**
     * Create a new command instance.
     *
     * @param AsnPresidioIntegration $presidioIntegrationService
     * @return void
     */
    public function __construct(AsnPresidioIntegration $presidioIntegrationService)
    {
        parent::__construct();
        $this->presidioIntegrationService = $presidioIntegrationService;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        if (!$this->presidioIntegrationService->checkConnection()) {
          
            $this->sendNotification(false);
            return $this->error("Connection failed");
        }

        $execStartTime  = Carbon::now();
        $dateFrom       = "2023-01-01";
        $dateTo         = Carbon::now()->format('Y-m-d');
        $loggedData     = CronJob::where('action', 'asn_presidio')->orderBy('created_at', 'desc')->first();
       
        if ($loggedData) {
          
            $dateFrom = Carbon::createFromFormat("Y-m-d H:i:s", $loggedData->date_to)->format('Y-m-d');
        }

        $pageNumber = 1;
        $totalImported = 0;
      
        try {
            while (1) {
           
                $response = $this->presidioIntegrationService->getOrders($dateFrom, $dateTo, $pageNumber++);
                
                if ($this->hasErrorInResponse($response)) {
                   
                    break;
                }

                $totalImported += $this->presidioIntegrationService->importPresidioDetails($response);
            };

            $this->updateImportLog($dateFrom, $execStartTime);
            $this->sendNotification(true);

            return $this->info($totalImported . " Assets imported");
        } catch (Exception $e) {
          
            Log::channel('daily')->info("Presidio API sync error :- " . $e->getMessage());
            $this->sendNotification(false);

            return $this->error("Failed to import data. Please try later.");
        }
    }

    /**
     * Update the cron table with execution time details
     *
     * @param Date $dateFrom         import data start date
     * @param Date $execStartTime    Import start time
     */
    public function updateImportLog($dateFrom, $execStartTime)
    {
        $cronData = [
            'action'        => 'asn_presidio',
            'started_at'    => $execStartTime,
            'ended_at'      => Carbon::now(),
            'date_from'     => $dateFrom,
            'date_to'       => $execStartTime,
        ];

        CronJob::create($cronData);
    }

    /**
     * Checking the API reponse has the error or not.
     * @param array $response
     * @return boolean
     */
    public function hasErrorInResponse($response)
    {
        if (!$response) {
            return true;
        }

        if (isset($response['status']) && $response['status'] == 404) {
            return true;
        }

        return false;
    }

    /**
     * Send slack notification
     *
     * @param bool $status
     */
    public function sendNotification(bool $status)
    {
        $message = $status ? 'ASN Presidio Sync completed.' : 'ASN Presidio Sync failed.';
     
        if (config('error-notification-mails.slack')) {

            Notification::route('slack', config('error-notification-mails.slack'))->notify(new Slack($message));
        }
    }
}
