<?php

namespace App\Console\Commands;


use Illuminate\Console\Command;
use Exception;
use App\Models\Asset;
use App\Models\AssetType;
use App\Models\AssetStatus;
use App\Models\Manufacturer;
use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;
use App\User;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Illuminate\Support\Facades\Storage;
use App\Events\BulkUpdates;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;


class ParseExcelFile extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'ParseExcel:run';


    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'One time runnable commands for quick fixes / updates on existing data';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }


    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $file = 'inventory.xlsx';
        $filePath = storage_path('app/public/' . $file);
        // Check if the file exists in the storage
        if (!Storage::disk('public')->exists($file)) {
            $this->error("File not found.");
            return;
        }
        // Load the Excel file using PhpSpreadsheet
        $spreadsheet = IOFactory::load($filePath);
        $this->createAssets($spreadsheet);
    } // end handle


    /**
     * Import inventory from sheet
     * @param mixed $spreadsheet
     * 
     * @return [type]
     */
    private function createAssets($spreadsheet)
    {
        $sheet = $spreadsheet->getSheetByName('Sheet1');
        $this->info("Processing Sheet: Sheet1");

        $assigned = AssetStatus::where('slug', 'assigned')->first()->id;
        $assetTypeId = AssetType::where('name', 'Laptop')->first()->id;

        $manufacturerId = Manufacturer::firstOrCreate([
            'name' => '-',
            'slug' => '-',
        ])->id;

        $makeAndModelId = MakeAndModel::firstOrCreate([
            'manufacturer_id' => $manufacturerId,
            'name' => '-',
            'slug' => '-',
            'asset_type_id' => $assetTypeId,
        ])->id;

        $technicalSpecsId = TechnicalSpecs::firstOrCreate([
            'details'  => '-',
            'make_and_model_id' => $makeAndModelId
        ])->id;

        // Get the highest row and column indices (the dimension of the data)
        $highestRow = $sheet->getHighestDataRow();
        // Loop through each row and create users
        for ($row = 2; $row <= $highestRow; ++$row) {
            try {

                $data['serial_no'] = $sheet->getCell('A' . $row)->getValue();

                if (empty($data['serial_no'])) {
                    continue;
                }
                $asset = Asset::where('serial_no', $data['serial_no'])->first();

                if ($asset) {
                    $this->info("Asset already exists: " . $sheet->getCell('A' . $row)->getValue());
                    continue;
                }
                $email = $sheet->getCell('G' . $row)->getValue();
                $user = User::where('email', $email)->first();
                $data['user_id'] = $user ? $user->id : 0;

                if ($data['user_id'] == 0) {
                    Log::channel('daily')->info("User not found: for Serial No: " . $data['serial_no'] . " and Email: " . $email);
                    continue;
                }

                $data['asset_tag'] = $sheet->getCell('B' . $row)->getValue();
                $data['asset_status_id'] = $assigned;
                $data['asset_type_id'] = $assetTypeId;
                $data['make_and_model_id'] = $makeAndModelId;
                $data['technical_spec_id'] = $technicalSpecsId;
                $asset = Asset::create($data);
                $this->createAssetHistory($asset);
            } catch (Exception $e) {
                $this->error($e->getMessage());
                $this->error("Error on inventory: " . $sheet->getCell('A' . $row)->getValue());
                continue;
            }
        }

        $this->info("Sheet: Inventory processed successfully!");
        return true;
    } // end createAssets


    /**
     * Create the asset history for a given asset.
     *
     * @param \App\Models\Asset $asset The asset for which to create the history.
     * @return void
     */
    public function createAssetHistory($asset)
    {
        $assigned = Cache::remember('assignedAssetStatus', 3600, function () {
            return AssetStatus::where('slug', 'assigned')->first();
        });
        $assetHistory = [
            'asset_id' => $asset->id,
            'action' => 'created',
            'new_asset_status_id' => $assigned->id,
            'new_value' => "Assigned",
            'description' => 'Created Asset <a href="/assets/' . $asset->id . ' target="__blank">' . $asset->serial_no . '</a> and assigned to ' . optional($asset->user)->email,
            'created_by' => 'Teqtivity Import',
        ];

        event(new BulkUpdates($assetHistory));
    } // end createAssetHistory


}
