<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use Illuminate\Console\Command;
use App\Services\Integrations\OktaDirectoryIntegration;
use Carbon\Carbon;
use App\Models\CronJob;
use Illuminate\Support\Facades\Notification;
use App\Notifications\OktaDirectoryImportSuccessfull;

class OktaDirectory extends Command
{

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'OktaDirectory:oktadirectory';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fetch data from Okta directory';


    /**
     * Create a new command instance.
     *
     * @param OktaDirectoryIntegration $oktaDirectoryIntegration An instance of OktaDirectoryIntegration for Okta services integration.
     *
     * @return void
     */
    public function __construct(protected OktaDirectoryIntegration $oktaDirectoryIntegration)
    {
        parent::__construct();
    }


    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $startTime = Carbon::now()->format('Y-m-d H:i:s');
        $count = 0;
        $dateTo = Carbon::now();
        $dateFrom = $this->getLoggedData();

        $this->info("User import from API started.");
        $count = $this->oktaDirectoryIntegration->importOktaDirectoryData($dateFrom, $dateTo);

        if ($count === null) {
            $message = 'FAILED. Okta Directory import failed.';
            $this->sendNotification($message);
            return $this->error($message);
        }

        $message = 'Okta Directory import completed. Total ' . $count . ' users updated.';

        $this->updateCronJob($startTime, $dateFrom, $dateTo);

        // Event to add sync history(sync_name, sync_status, notes).
        event(new IntegrationSyncHistory('okta_directory'));

        $this->sendNotification($message);

        return $this->info($message);
    }


    /**
     * Get the last updated date
     *
     * @return \Carbon\Carbon The date from which to start the next log data collection.
     */
    public function getLoggedData()
    {
        $dateFrom = Carbon::now()->subDay();

        $loggedData = CronJob::where('action', 'okta_directory')->orderBy('created_at', 'desc')->first();

        if ($loggedData) {
            $dateFrom = Carbon::parse($loggedData->date_to);
        }

        return $dateFrom;
    }

    /**
     * Update current api call details in the cron table
     *
     * @param \Carbon\Carbon $startTime The start time of the cron job.
     * @param string         $dateFrom  The starting date of the data processed by the cron job.
     * @param string         $dateTo    The ending date of the data processed by the cron job.
     *
     * @return void
     */
    public function updateCronJob($startTime, $dateFrom, $dateTo)
    {
        $cronData = [
            'action' => 'okta_directory',
            'started_at' => $startTime,
            'ended_at' => Carbon::now(),
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];

        CronJob::create($cronData);
    }

    /**
     * Send a Slack notification with the given message.
     * 
     * @param string $message The message content to be sent as a notification.
     * 
     * @return void
     */
    public function sendNotification($message)
    {
        if (config('teqtivity-notification-mails.slack')) {
            Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new OktaDirectoryImportSuccessfull($message));
        }
    }
}
