<?php

namespace App\Console\Commands;

use App\Models\SlackNotificationCredential;
use App\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use App\Notifications\NewHiresWithZeroAssetsNotification;

class NewHiresWithZeroAssets extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'NewHiresWithZeroAssets:notify';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sending notification with the list of new hires with zero assets.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        try {
            $webhook = SlackNotificationCredential::where('slug', 'it_channel')->first();

            if (!$webhook) {
                return $this->info('Invalid web hook url');
                Log::error('NewHiresWithZeroAssets : Invalid web hook url');
            }

            $currentDate = Carbon::now()->toDateString();
            $newHiresWithAssets =  User::superAdmin()
                        ->doesntHave('assets')
                        ->whereDate('hire_date', '>=', $currentDate)
                        ->pluck('email')
                        ->toArray();

            if (!empty($newHiresWithAssets)) {
                $message = implode(", ", $newHiresWithAssets);
                Notification::route('slack', $webhook->webhook_url)
                    ->notify(new NewHiresWithZeroAssetsNotification($message));
                
                return $this->info(json_encode([
                    'status'  => 'success',
                    'message' => 'Notifications Send successfully'
                ]));
            }

            return $this->info(json_encode([
                'status'  => 'success',
                'message' => 'Nothing to notify'
            ]));

        } catch (Exception $e) {
            return $this->info(json_encode([
                'status'  => 'error',
                'message' => 'Something went wrong. Try again later'
            ]));
            Log::error('NewHiresWithZeroAssets : ' . $e->getMessage());
        }
    }
}
