<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use Illuminate\Console\Command;
use App\Services\Integrations\MobileIronApi;
use App\Models\MobileIron;
use App\Notifications\Slack;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Log;

class MobileIronUpload extends Command
{

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'MobileIronUpload:upload';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Upload the MobileIron API data';


    /**
     * Create a new command instance.
     *
     * @param MobileIronApi $service The MobileIron API service instance.
     *
     * @return void
     */
    public function __construct(protected MobileIronApi $service)
    {
        parent::__construct();
    }


    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        if (!$this->service->checkApiConnection()) {
            // Send api filed message.
            $this->info("Connection Failed !");
            $this->sendNotification('FAILED! MobileIron connection failed.');

            return false;
        }

        $this->info('Start time ' . date('Y-m-d H:i:s'));
        Log::channel('single')->info("MobileIron data upload started.");

        $this->info("MobileIron data upload started.");

        MobileIron::truncate();

        $this->service->upload();
        $this->autoResassignDiscrepacyDevices();

        Log::channel('single')->info("MobileIron data upload finished.");

        $this->info('End time ' . date('Y-m-d H:i:s'));
        $message = 'MobileIron data upload finished.';

        $this->sendNotification($message);

        // Event to add sync history(sync_name, sync_status, notes).
        event(new IntegrationSyncHistory('mobile-iron'));

        return true;
    }


    /**
     * Re assign the user for user mismatch devices.
     *
     * @return mixed
     */
    private function autoResassignDiscrepacyDevices()
    {
        return $this->service->reassignAssets();
    }


    /**
     * Sends a notification message via Slack.
     *
     * @param string $message The message content to be sent via Slack.
     *
     * @return bool
     */
    public function sendNotification($message)
    {
        Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));

        return true;
    }
}
