<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use Illuminate\Console\Command;
use App\Models\KandjiDevices;
use App\Notifications\Slack;
use App\Services\Integrations\KandjiIntegration;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Log;

class KandjiSync extends Command
{

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'KandjiDevices:sync';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync the Kandji MDM devices';


    /**
     * Create a new command instance.
     *
     * @param KandjiIntegration $service The Kandji Integration service instance.
     *
     * @return void
     */
    public function __construct(protected KandjiIntegration $service)
    {
        parent::__construct();
    }


    /**
     * Execute the console command.
     *
     * @return bool True if the Kandji sync is successful, otherwise false.
     */
    public function handle()
    {
        if (!$this->service->checkApiConnection()) {
            // Send api filed message.
            $this->sendNotification('FAILED! Kandji connection failed.');

            return false;
        }

        $this->info('Start time ' . date('Y-m-d H:i:s'));
        Log::channel('single')->info("Kandji data upload started.");
        $this->info("Kandji data upload started");

        KandjiDevices::truncate();

        $this->service->upload();

        Log::channel('single')->info("Kandji data upload finished.");
        $this->info('End time ' . date('Y-m-d H:i:s'));

        $assetCount = KandjiDevices::count();

        $message = 'Kandji sync Completed. Total ' . $assetCount . ' devices added.';

        $this->sendNotification($message);

        //event to add sync history(sync_name, sync_status, notes)
        event(new IntegrationSyncHistory('kandji'));

        return true;
    }


    /**
     * Sends a notification message via Slack.
     *
     * @param string $message The message content to be sent via Slack.
     *
     * @return bool
     */
    public function sendNotification($message)
    {
        Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));

        return true;
    }
}
