<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\JumpCloudDevices;
use App\Notifications\Slack;
use App\Services\DiscoveryTools\JumpCloud\JumpCloudSyncService;
use App\Services\Integrations\JumpCloudIntegration;
use Illuminate\Support\Facades\Notification;

class JumpCloudSync extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'jumpCloud:sync';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync the JumpCloud MDM devices';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(protected JumpCloudIntegration $apiService, protected JumpCloudSyncService $syncService)
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        if ($this->apiService->checkApiConnection() === false) {
            $this->sendNotification('FAILED! JumpCloud connection failed.');
            return false;
        }

        $this->info("JumpCloud data upload started at " . date('Y-m-d H:i:s'));
        JumpCloudDevices::truncate();
        $this->syncService->import();

        //auto create & assign assets if not exist
        $this->syncService->createAndAssignAssets();


        //auto reassign discrepancy devices
        // $this->syncService->autoResassignAssets();


        $this->info('End time ' . date('Y-m-d H:i:s'));

        $assetCount = JumpCloudDevices::count();
        $message = 'JumpCloud sync completed. Total ' . $assetCount . ' devices added.';
        $this->sendNotification($message);

        return $this->info($message);
    }

    /**
     * Send Mail Notification
     *
     * @param mixed $message
     */
    public function sendNotification($message)
    {
        if (config('teqtivity-notification-mails.slack')) {
            Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));
        }

        return true;
    }
}
