<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use App\Models\JamfComputer;
use App\Services\Integrations\JamfIntegration;
use App\Services\DiscoveryTools\Jamf\JamfDataSyncService;
use App\Services\DiscoveryTools\Jamf\JamfMobileDataSyncService;
use Illuminate\Console\Command;
use App\Notifications\Slack;
use Exception;
use Illuminate\Support\Facades\Notification;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\Log;

class JamfDataSync extends Command
{

    const PAGE_SIZE = 100;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'JamfDataSync:run';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Download data from Jamf Pro API';


    /**
     * Create a new command instance.
     *
     * @param JamfIntegration           $jamfApiService            Handles the basic API integration with the JAMF system.
     * @param JamfDataSyncService       $jamfDataSyncService       Manages synchronization of general data from JAMF.
     * @param JamfMobileDataSyncService $jamfMobileDataSyncService Manages synchronization of mobile-specific data from JAMF.
     *
     * @return void
     */
    public function __construct(protected JamfIntegration $jamfApiService, protected JamfDataSyncService $jamfDataSyncService, protected JamfMobileDataSyncService $jamfMobileDataSyncService)
    {
        parent::__construct();
    }


    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        //For Checking Connection status
        if (!$this->jamfApiService->checkConnection()) {
            $this->sendNotification('FAILED! Jamf connection failed.');
            return $this->info("exiting from execution");
        }

        JamfComputer::truncate();

        $assetCount = 0;

        $this->syncComputerDevices();

        // $this->syncMobileDevices();

        // $this->syncPreStageComputerDevices();

        $assetCount = JamfComputer::count();

        $message = 'Jamf sync Completed. Total ' . $assetCount . ' devices added';

        // $this->autoResassignDiscrepacyDevices();

        $this->sendNotification($message);

        // Event to add sync history(sync_name, sync_status, notes).
        event(new IntegrationSyncHistory('jamf'));

        return $this->info($message);
    }


    /**
     * Paginated sync of computer devices
     *
     * @return int The total number of synchronized computer devices.
     */
    public function syncComputerDevices()
    {
        $page = 0;
        $assetCount = 0;

        do {
            try {
                $data = $this->jamfApiService->getJamfProComputers($page, self::PAGE_SIZE);
                $count = $this->saveData($data->results);
                $page++;
                $assetCount += $count;
            } catch (\Exception $e) {
                Log::channel('daily')->error("Jamf Data Sync Main catch on page " . $page . PHP_EOL);
                $page++;
            }
        } while ($page < $data->totalCount / self::PAGE_SIZE);

        return $assetCount;
    }


    /**
     * Saves the data for each item in the given array.
     *
     * @param array $items The array of items to save data for.
     *
     * @return int The number of items that were successfully saved.
     */
    public function saveData($items)
    {
        $count = 0;

        foreach ($items as $item) {
            $deviceDetails = $this->jamfApiService->getProDeviceDetailsById($item->id);
            $this->jamfDataSyncService->updateJamfData($item, $deviceDetails);
            $count++;
        }

        return $count;
    }


    /**
     * Sync the Mobile devices
     *
     * @return bool Indicates whether the synchronization process was successful.
     */
    public function syncMobileDevices()
    {
        $page = 0;

        do {
            $data = $this->jamfApiService->getMobileDevices($page);
            $this->saveMobileData($data->results);
            $page++;
        } while ($page < $data->totalCount / self::PAGE_SIZE);

        return true;
    }


    /**
     * Save mobile Devices
     *
     * @param array $items The array of mobile devices to save.
     *
     * @return bool Indicates whether the mobile device data was successfully saved.
     */
    public function saveMobileData($items)
    {
        foreach ($items as $item) {
            try {
                $deviceDetails = $this->jamfApiService->getMobileDetailsById($item->id);

                $this->jamfMobileDataSyncService->updateJamfData($item, $deviceDetails);
            } catch (Exception $e) {
                Log::error("Jamf Mobile Sync Error - " . $item->serialNumber . " -- " . $e->getMessage());
            }
        }

        return true;
    }


    /**
     * Sync the Pre Stage Computer Devices
     *
     * @return bool Always returns true, indicating the process has completed.
     */
    public function syncPreStageComputerDevices()
    {
        $computerPreStages = $this->jamfApiService->getComputerPrestages();
        foreach ($computerPreStages->results as $key => $preStages) {
            $computerPreStagesDevices = $this->jamfApiService->getComputerPreStagesDevices($preStages->id);

            foreach ($computerPreStagesDevices->assignments as $key => $item) {
                try {
                    $this->jamfApiService->getComputerDetailsBySerialNo($item->serialNumber);
                } catch (ClientException $e) {
                    // If there no asset details found when search by serial no, it will throw ClientException error.
                    // If serials found in the stages API and we not getting the serials, we assume that this are the staged device.
                    // So we will create this asset to the SFO storage.
                    $this->jamfDataSyncService->createPrestageDevice($item->serialNumber);
                } catch (Exception $e) {
                    $this->info("Computer Pre Stage Sync Error == ." . $item->serialNumber);
                    Log::error("Jamf Computer Pre Stage Sync Error - " . $item->serialNumber . " -- " . $e->getMessage());
                }
            }
        }
        return true;
    }


    /**
     * Re assign the user for user mismatch devices.
     *
     * @return mixed Returns the result of the auto-reassignment process.
     */
    private function autoResassignDiscrepacyDevices()
    {
        return $this->jamfDataSyncService->autoReassignAssets();
    }


    /**
     * Sends a notification message via Slack.
     *
     * @param string $message The message content to be sent via Slack.
     *
     * @return void
     */
    public function sendNotification($message)
    {
        Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));
    }
}
