<?php

namespace App\Console\Commands;

use App\Notifications\Slack;
use App\Services\GoogleWorkspace\GsuiteLocationService;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Notification;

class GsuiteLocationsSync extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'GsuiteLocations:gsuitelocations';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import locations from gsuite';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct(protected GsuiteLocationService $service)
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $locationCount = 0;
        $client = $this->service->getJwtAccessToken();
        if (!$client) {
            $this->sendNotification('FAILED. Gsuite locations connection failed.');
            return $this->info("Gsuite locations connection failed.");
        }
        try {
            $locationCount  = $this->service->importGsuiteLocations();

            if ($locationCount === null) {
                $this->sendNotification('FAILED. Gsuite locations sync failed.');

                return $this->info("Gsuite locations sync failed.");
            }

            $message = "Gsuite locations sync completed. Total  $locationCount  locations added.";
            $this->sendNotification($message);

            return $this->info($message);
        } catch (Exception $e) {
            // dd($e->getMessage(), $e->getLine(), $e->getFile());
            $this->sendNotification('FAILED. Gsuite locations connection failed.');

            return $this->info("Gsuite location sync failed.");
        }
    }

    /**
     * Send Mail Notification
     *
     * @param mixed $message
     */
    public function sendNotification($message)
    {
        if (config('teqtivity-notification-mails.slack')) {
            Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));
        }

        return true;
    }
}
