<?php

namespace App\Console\Commands;

use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CopyTableContents extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'copyAllTables:run';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'To copy data from all tables in the source database to the corresponding tables in the destination database';


    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        // Establish connections to both databases.
        $sourceConnection = DB::connection('teqtivity_milestone');
        $destinationConnection = DB::connection('tqt_190_empty_db');

        // Get the list of tables from the source database.
        $tables = $sourceConnection->getDoctrineSchemaManager()->listTableNames();

        //Iterate over each table
        foreach ($tables as $tableName) {
            // Check if the table exists in both databases.
            if ($tableName == 'migrations') {
                continue;
            }

            if ($sourceConnection->getSchemaBuilder()->hasTable($tableName) && $destinationConnection->getSchemaBuilder()->hasTable($tableName)) {
                // Retrieve the data from the source table.
                try {
                    $insertCount = 0;

                    // Truncate the destination table to remove existing data.
                    $destinationConnection->statement('SET FOREIGN_KEY_CHECKS=0');
                    $destinationConnection->table($tableName)->truncate();

                    // Get the columns that exist in both source and destination tables.
                    $commonColumns = array_intersect(
                        $sourceConnection->getSchemaBuilder()->getColumnListing($tableName),
                        $destinationConnection->getSchemaBuilder()->getColumnListing($tableName)
                    );

                    $query = $sourceConnection->table($tableName)->select($commonColumns);

                    if ($tableName == 'user_types') {
                        $destinationConnection->statement('ALTER TABLE user_types MODIFY COLUMN deleted_at DATETIME DEFAULT NULL;');
                    }

                    // Check if the table has an 'id' column.
                    if ($sourceConnection->getSchemaBuilder()->hasColumn($tableName, 'id')) {
                        $query->orderBy('id');

                        $query->chunk(1000, function ($sourceData) use ($tableName, $destinationConnection, &$insertCount) {
                            // Convert the collection of stdClass objects to an array.
                            $sourceArray = json_decode(json_encode($sourceData), true);

                            // Handle '0000-00-00 00:00:00' values and convert them to null.
                            foreach ($sourceArray as &$row) {
                                foreach ($row as $key => $value) {
                                    if ($value === '0000-00-00 00:00:00' || $value === '0000-00-00') {
                                        $row[$key] = null;
                                    }
                                }
                            }

                            // Copy the data from the source array to the destination table.
                            $destinationConnection->table($tableName)->insert($sourceArray);
                            $insertCount += count($sourceArray);
                        });
                    } else {
                        // Retrieve all the data from the source table.
                        $sourceData = $query->get();

                        // Handle '0000-00-00 00:00:00' values and convert them to null.
                        $sourceArray = json_decode(json_encode($sourceData), true);
                        foreach ($sourceArray as &$row) {
                            foreach ($row as $key => $value) {
                                if ($value === '0000-00-00 00:00:00') {
                                    $row[$key] = null;
                                }
                            }
                        }

                        // Copy the data from the source array to the destination table.
                        $destinationConnection->table($tableName)->insert($sourceArray);
                        $insertCount += count($sourceArray);
                    }

                    $destinationConnection->statement('SET FOREIGN_KEY_CHECKS=1');

                    $this->info("Inserted " . $insertCount . " rows into " . $tableName);
                } catch (\Throwable $e) {
                    $this->info("Error inserting on " . $tableName);
                    Log::channel('daily')->info($e->getMessage());

                    continue;
                }
            } else {
                $this->info("Destintion table doesnot exist - " . $tableName);
            }
        }

        $this->info('Data copied successfully.');
    }
}
