<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use App\Models\Chromebook;
use App\Notifications\Slack;
use Illuminate\Console\Command;
use App\Services\DiscoveryTools\Chromebook\ChromeBookSyncService;

use Illuminate\Support\Facades\Notification;
use Artisan;
use Exception;

class ChromebookSync extends Command
{

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'Chromebook:sync';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Upload the ChromebookS data';


    /**
     * Create a new command instance.
     *
     * @param ChromeBookSyncService $service       The ChromeBookSyncService instance.
     *
     * @return void
     */
    public function __construct(protected ChromeBookSyncService $service)
    {
        parent::__construct();
    }

    /**
     * Handle the ChromeBook synchronization process.
     *
     * @return void
     */
    public function handle()
    {

        $assetCount = 0;
        $client = $this->service->getJwtAccessToken();

        if (!$client) {
            $this->sendNotification('FAILED. Chromebook connection failed.');
            return $this->info("ChromeBook connection failed.");
        }

        try {

            // Chromebook::truncate();

            $assetCount  = $this->service->importChromeOsdevices();

            if ($assetCount === null) {
                $this->sendNotification('FAILED.Chromebook sync failed.');

                return $this->info("Chromebook sync failed.");
            }

            $message = "Chromebook sync completed. Total  $assetCount  devices added.";

            $this->autoResassignDiscrepacyDevices();

            // Event to add sync history(sync_name, sync_status, notes).

            event(new IntegrationSyncHistory('chromebook'));

            $this->sendNotification($message);

            return $this->info($message);
        } catch (Exception $e) {
            // dd($e->getMessage());
            $this->sendNotification('FAILED. Chromebook connection failed.');

            return $this->info("ChromeBook sync failed.");
        }
    }

    /**
     * Re assign the user for user mismatch devices.
     *
     * @return bool The result of the reassignment operation
     */
    private function autoResassignDiscrepacyDevices()
    {
        return $this->service->reassignAssets();
    }

    /**
     * Sends a notification message.
     *
     * @param string $message The message to be sent.
     *
     * @return bool True if the notification was sent successfully, false otherwise.
     */
    public function sendNotification($message)
    {
        if (config('teqtivity-notification-mails.slack')) {
            Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));
        }

        return true;
    }
}
