<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use App\Models\CloudAsset;
use App\Notifications\Slack;
use App\Services\CloudAssets\Aws\AwsCloudAssetsSyncService;
use App\Services\Integrations\CloudAssets\AwsIntegration;
use Illuminate\Console\Command;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class AwsCloudAssetsSync extends Command
{

    /**
     * Constructor for the class.
     *
     * @param AwsIntegration            $apiService description
     * @param AwsCloudAssetsSyncService $service    description
     *
     * @return void
     */
    public function __construct(protected AwsIntegration $apiService, protected AwsCloudAssetsSyncService $service)
    {
        parent::__construct();
    }

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'AwsInstanceSync:run';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync AWS Instances using Ec2Client API';


    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        try {
            $connection = $this->apiService->checkApiConnection();

            if ($connection === false) {
                $this->sendNotification('FAILED! AWS cloud assets sync failed.');

                return;
            }

            CloudAsset::truncate();

            $this->syncInstance();
            $this->syncS3Buckets();
            $this->SyncRds();

            $syncCount = CloudAsset::count();

            // Send Notification
            $this->sendNotification('AWS cloud assets sync completed. Total ' . $syncCount . 'cloud assets added');

            // Event to add sync history(sync_name, sync_status, notes).
            event(new IntegrationSyncHistory('aws-cloud-assets'));
        } catch (Exception $e) {
            $this->sendNotification('FAILED! AWS cloud assets sync failed.');
            Log::error($e->getMessage());
        }
    }

    /**
     * Synchronizes Instances buckets with API from multiple regions.
     *
     * @return void
     */
    public function syncInstance(): void
    {
        $apiCredentials = $this->apiService->getCredentials();
        $regions = array_filter(explode(",", $apiCredentials->custom_data));

        foreach ($regions as $region) {
            $apiResponse = $this->apiService->getInstances($region);

            if ($apiResponse) {
                $this->service->saveInstances($apiResponse, $region);
            }
        }
    }

    /**
     * Synchronizes S3 buckets with API from a defined region.
     *
     * @return void
     */
    public function syncS3Buckets(): void
    {
        // Defining a default region for S3Client.
        $region = 'us-east-2';

        $apiResponse = $this->apiService->getS3Buckets($region);

        if ($apiResponse) {
            $this->service->saveS3Buckets($apiResponse, $region);
        }
    }

    /**
     * Synchronizes RDS buckets with API from multiple regions.
     *
     * @return void
     */
    public function SyncRds(): void
    {
        $apiCredentials = $this->apiService->getCredentials();
        $regions = array_filter(explode(",", $apiCredentials->custom_data));

        foreach ($regions as $region) {
            $apiResponse = $this->apiService->getRds($region);

            if ($apiResponse) {
                $this->service->saveRds($apiResponse, $region);
            }
        }
    }

    /**
     * Sends a notification based on the configured methods.
     *
     * @param mixed $message
     */
    public function sendNotification($message)
    {
        if (config('teqtivity-notification-mails.slack')) {
            Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));
        }
    }
}
