<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\Integrations\ShipmentTracking\FedexTracking;
use App\Services\Integrations\ShipmentTracking\UpsTracking;
use App\Models\AssetTracking;
use App\Notifications\AssetStatusSyncNotification;
use Illuminate\Support\Facades\Notification;

class AssetTrackingSync extends Command
{

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'AssetTracking:sync';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update the asset\'s tracking status from fedex and ups';


    /**
     * Create a new command instance.
     *
     * @param  FedexTracking $fedexService The service for FedEx tracking.
     * @param  UpsTracking   $upsService   The service for UPS tracking.
     *
     * @return void
     */
    public function __construct(protected FedexTracking $fedexService, protected UpsTracking $upsService)
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle()
    {
        $assetTrackings = AssetTracking::where('active', 1)->get();

        foreach ($assetTrackings as $assetTracking) {
            $this->updateTrackingStatus($assetTracking);
        }

        $message = 'Sync Completed. Total ' . count($assetTrackings) . ' asset statuses updated';

        Notification::route('slack', config('teqtivity-notification-mails.slack'))
            ->notify(new AssetStatusSyncNotification($message));

        $this->info($message);
    }

    /**
     * Update the tracking status of an asset.
     *
     * This method fetches tracking details for the given tracking number
     * and updates the asset's tracking information if details are available.
     *
     * @param AssetTracking $assetTracking The asset tracking information to update.
     *
     * @return void
     */
    public function updateTrackingStatus($assetTracking)
    {
        if (isUpsTrackingNo($assetTracking->tracking_number)) {
            $trackingDetails = $this->upsService->getTrackingDetails($assetTracking->tracking_number);
        } else {
            $trackingDetails = $this->fedexService->getTrackingDetails($assetTracking->tracking_number);
        }

        if ($trackingDetails) {
            $dataToUpdate = $this->getDataToUpdate($trackingDetails);

            if ($dataToUpdate) {
                $assetTracking->update($dataToUpdate);
            }
        }
    }

    /**
     * Get the data to update for the asset tracking information based on tracking details.
     *
     * @param array $trackingDetails The tracking details obtained for the asset.
     *
     * @return array|null The data to update for the asset tracking, or null if no update is required.
     */
    public function getDataToUpdate($trackingDetails)
    {
        if (in_array($trackingDetails['status'], ['ERROR', 'No tracking information available'])) {
            return ['active' => 0];
        }

        $dataToUpdate = [
            'shipment_status'   => $trackingDetails['status'],
            'status_updated_on' => $trackingDetails['updated_on']
        ];

        if ($trackingDetails['status'] == 'Delivered') {
            $dataToUpdate['active'] = 0;
        }

        return $dataToUpdate;
    }
}
